//*********************************************************************************
//**
//** Project.........: Magnetic Loop Controller
//**
//**
//**
//** Platform........: AT90usb1286 @ 16MHz
//**
//** Licence.........: This software is freely available for non-commercial 
//**                   use only - i.e. for research and experimentation.
//**
//** Initial version.: 2012-10-20, Loftur Jonasson, TF3LJ / VE2LJX
//**
//** Current version.: See ML.h
//**
//** History.........: ...
//**
//*********************************************************************************

//
// This code makes use of the LUFA library for USB connectivity
//
/*
             LUFA Library
     Copyright (C) Dean Camera, 2012.

  dean [at] fourwalledcubicle [dot] com
           www.lufa-lib.org
*/

/*
  Copyright 2012  Dean Camera (dean [at] fourwalledcubicle [dot] com)

  Permission to use, copy, modify, distribute, and sell this
  software and its documentation for any purpose is hereby granted
  without fee, provided that the above copyright notice appear in
  all copies and that both that the copyright notice and this
  permission notice and warranty disclaimer appear in supporting
  documentation, and that the name of the author not be used in
  advertising or publicity pertaining to distribution of the
  software without specific, written prior permission.

  The author disclaim all warranties with regard to this
  software, including all implied warranties of merchantability
  and fitness.  In no event shall the author be liable for any
  special, indirect or consequential damages or any damages
  whatsoever resulting from loss of use, data or profits, whether
  in an action of contract, negligence or other tortious action,
  arising out of or in connection with the use or performance of
  this software.
*/


//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
// Microcontroller Pin assignment (further defined in the PM.h file):
//
// PB0 = 1st A4975 D0
// PB1 = 1st A4975 D1
// PB2 = 1st A4975 D2
// PB3 = 1st A4975 Phase
// PB4 = 2nd A4975 D0
// PB5 = 2nd A4975 D1
// PB6 = 2nd A4975 D2
// PB7 = 2nd A4975 Phase
//
// PC0 = LCD control RS
// PC1 = LCD control RW
// PC2 = LCD control E
// PC3
// PC4 = LCD D4
// PC5 = LCD D5
// PC6 = LCD D6
// PC7 = LCD D7
//
// PD2 = USART RXD
// PD3 = USART TXD (unused)
//
// PD4 = Push Button Selector input
// PD5 = Rotary Encoder A input
// PD6 = LED output 
// PD7 = Rotary Encoder B input
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------
//----------------------------------------------------------------------------


#include "ML.h"
#include "USB/USBtoSerial.h"				// LUFA include
#include "analog.h"
#include <math.h>

EEMEM		var_t	E;						// Variables in eeprom (user modifiable, stored)
			var_t	R						// Variables in ram/flash rom (default)
				=
				{
						COLDSTART_REF		// Update into eeprom if value mismatch
					//, 	ENC_RES_DIVIDE		// Initial Encoder Resolution
					, { 0					// Calibrate Point 1  (0 = uncalibrated)
					, 0						// Calibrate Point 2  (0 = uncalibrated)
					, 0						// Calibrate Point 3  (0 = uncalibrated)
					, 0						// Calibrate Point 4  (0 = uncalibrated)
					, 0						// Calibrate Point 5  (0 = uncalibrated)
					, 0						// Calibrate Point 6  (0 = uncalibrated)
					, 0						// Calibrate Point 7  (0 = uncalibrated)
					, 0						// Calibrate Point 8  (0 = uncalibrated)
					, 0						// Calibrate Point 9  (0 = uncalibrated)
					, 0						// Calibrate Point 10 (0 = uncalibrated)
					, 0              		// Calibrate Point 11 (0 = uncalibrated)
					, 0						// Calibrate Point 12 (0 = uncalibrated)
					, 0						// Calibrate Point 13 (0 = uncalibrated)
					, 0						// Calibrate Point 14 (0 = uncalibrated)
					, 0						// Calibrate Point 15 (0 = uncalibrated)
					, 0						// Calibrate Point 16 (0 = uncalibrated)
					, 0						// Calibrate Point 17 (0 = uncalibrated)
					, 0						// Calibrate Point 18 (0 = uncalibrated)
					, 0						// Calibrate Point 19 (0 = uncalibrated)
					, 0						// Calibrate Point 20 (0 = uncalibrated)
					, 0              		// Calibrate Point 21 (0 = uncalibrated)
					, 0						// Calibrate Point 22 (0 = uncalibrated)
					, 0						// Calibrate Point 23 (0 = uncalibrated)
					, 0						// Calibrate Point 24 (0 = uncalibrated)
					, 0						// Calibrate Point 25 (0 = uncalibrated)
					, 0						// Calibrate Point 26 (0 = uncalibrated)
					, 0						// Calibrate Point 27 (0 = uncalibrated)
					, 0						// Calibrate Point 28 (0 = uncalibrated)
					, 0						// Calibrate Point 29 (0 = uncalibrated)
					, 0 }					// Calibrate Point 30 (0 = uncalibrated)
											// Calibration points will be auto sorted from lowest to highest.
											// The stepper motor will not automatically tune by serial command
											// any further than the lowest/highest calibration points.
					, { 1000000				// Preset Position 1  (1000000 = uncalibrated)
					, 1000000				// Preset Position 2  (1000000 = uncalibrated)
					, 1000000				// Preset Position 3  (1000000 = uncalibrated)
					, 1000000				// Preset Position 4  (1000000 = uncalibrated)
					, 1000000				// Preset Position 5  (1000000 = uncalibrated)
					, 1000000				// Preset Position 6  (1000000 = uncalibrated)
					, 1000000				// Preset Position 7  (1000000 = uncalibrated)
					, 1000000				// Preset Position 8  (1000000 = uncalibrated)
					, 1000000				// Preset Position 9  (1000000 = uncalibrated)
					, 1000000				// Preset Position 10 (1000000 = uncalibrated)
					, 1000000				// Preset Position 11 (1000000 = uncalibrated)
					, 1000000				// Preset Position 12 (1000000 = uncalibrated)
					, 1000000				// Preset Position 13 (1000000 = uncalibrated)
					, 1000000				// Preset Position 14 (1000000 = uncalibrated)
					, 1000000				// Preset Position 15 (1000000 = uncalibrated)
					, 1000000				// Preset Position 16 (1000000 = uncalibrated)
					, 1000000				// Preset Position 17 (1000000 = uncalibrated)
					, 1000000				// Preset Position 18 (1000000 = uncalibrated)
					, 1000000				// Preset Position 19 (1000000 = uncalibrated)
					, 1000000				// Preset Position 20 (1000000 = uncalibrated)
					, 1000000				// Preset Position 21 (1000000 = uncalibrated)
					, 1000000				// Preset Position 22 (1000000 = uncalibrated)
					, 1000000				// Preset Position 23 (1000000 = uncalibrated)
					, 1000000				// Preset Position 24 (1000000 = uncalibrated)
					, 1000000				// Preset Position 25 (1000000 = uncalibrated)
					, 1000000				// Preset Position 26 (1000000 = uncalibrated)
					, 1000000				// Preset Position 27 (1000000 = uncalibrated)
					, 1000000				// Preset Position 28 (1000000 = uncalibrated)
					, 1000000				// Preset Position 29 (1000000 = uncalibrated)
					, 1000000 }				// Preset Position 30 (1000000 = uncalibrated)
											// Preset positions and calibration points go as pairs.
					, 14000000				// Current frequency in Hz (as auto saved)
					, 1000000				// Current stepper position (as auto saved)
											// Each revolution of the stepper motor (Nema17, 1.8deg per step)
											// is 1600 positions, including micro steps.  Typical variable
											// vacuum capacitor has 20 - 22 revolutions for full range.
											// A starting point of 1000000 is a good round number to comfortably
											// ensure that we don't hit any boundary conditions.

							};


uint8_t		Status = 0;						// Contains various status flags
uint8_t		Menu_Mode;						// Menu Mode Flags
uint8_t		Menu_disp_timer;				// Used for a timed display when returning from Menu

char lcd_buf[20];

//DEBUG
int32_t	stepper_track, stepper_start_pos, delta_Pos=0;
int32_t tunedFrq;


//
//-----------------------------------------------------------------------------------------
// Recalibrate Stepper Position if Short Push
//-----------------------------------------------------------------------------------------
//
void recalibrate_stepper_pos(void)
{
	uint8_t i;

	// Scan through all "active" memories
	for (i=0; i<NUM_PRESETS && R.presetFrq[i]>0; i++)
	{
		R.presetPos[i] = R.presetPos[i] + delta_Pos;

		eeprom_write_block(&R.presetPos[i], &E.presetPos[i], sizeof (R.presetPos[i]));
	}
	delta_Pos = 0;
}



//
//-----------------------------------------------------------------------------------------
// Track frequency and adjust stepper accordingly
//-----------------------------------------------------------------------------------------
//
void track_frq(void)
{
	int8_t i, range=0, num_active=0;//, below=0;
	int32_t delta_R, delta_F;
	double delta_calc;

	//
	// Find highest active preset
	//
	for (i=0; i<NUM_PRESETS; i++)
	{
		if (R.presetFrq[i] == 0)	// We're beyond the highest active memory
		{
			num_active = i;
			break;
		}
	}

	//
	// Protect capacitor from damage in case of misbehavior by software by
	// blocking automatic tuning if we are outside of stored presets
	//
	// Indicate if frequency is outside of the range of configured presets
	if((R.presetFrq[0]==0) || (R.curFrq < R.presetFrq[0]) || (R.curFrq > R.presetFrq[num_active -1]))
	{
		if (Menu_disp_timer == 0)					// Menu functions have precedence
		{
			lcdGotoXY(0,3);
			lcdPrintData("FRQ outside of range",20);
		}
	}
	// Indicate if capacitor position is outside of preset range 
	else if(((stepper_track-10 > R.presetPos[0]) && (stepper_track-10 > R.presetPos[num_active -1])) ||
			((stepper_track+10 < R.presetPos[0]) && (stepper_track+10 < R.presetPos[num_active -1])))
	{
		if (Menu_disp_timer == 0)					// Menu functions have precedence
		{
			lcdGotoXY(0,3);
			lcdPrintData("CAP outside of range",20);
		}
	}
		
	// Inside presets, It is OK to do stuff
	else
	{
		//
		// Calculate Stepper position to tune to, based on indicated frequency
		//
		// Find preset immediately above current frequency
		// Returns 0 if frequency is lower than lowest or
		// higher than highest reference frequency
		for (i=0; i<num_active; i++)
		{
			if (R.curFrq <= R.presetFrq[i])
			{
				range = i;
				break;
			}
		}
		// deltas used to calculate between Stepper Position and Frequency
		delta_R = R.presetPos[range] - R.presetPos[range-1];
		delta_F = R.presetFrq[range] - R.presetFrq[range-1];
		// calculate Stepper Position
		delta_calc = (double)  delta_R / delta_F;
		delta_calc = delta_calc * (R.curFrq - R.presetFrq[range-1]);
		R.curPos   = (int32_t) delta_calc + R.presetPos[range-1];
	
		//
		// Calculate Tuned Frequency based on current stepper position
		//
		// Find preset immediately above current stepper position
		// Returns 0 if position is lower than lowest or
		// higher than highest stored reference
		int8_t sign;		// Normalise direction of travel
		if (R.presetPos[1]>R.presetPos[0]) sign = 1;
		else sign = -1;
		for (i=0; i<num_active; i++)
		{
			if ((stepper_track*sign) <= (R.presetPos[i]*sign))
			{
				range = i;
				break;
			}
		}
		// deltas used to calculate between Stepper Position and Frequency
		delta_R = R.presetPos[range] - R.presetPos[range-1];
		delta_F = R.presetFrq[range] - R.presetFrq[range-1];
		// calculate Tuned Frequency
		delta_calc = (double)  delta_F / delta_R;
		delta_calc = delta_calc * (stepper_track - R.presetPos[range-1]);
		tunedFrq   = (int32_t) delta_calc + R.presetFrq[range-1];
	
		//
		// Position the Stepper according to Frequency
		//
		// Adjust stepper
		if(stepper_track < (R.curPos + delta_Pos))
		{
			stepper_Incr();	
			stepper_track++;
			Status |= FRQ_TIMER;				// New frequency, these flags seed timer to monitor if
			Status |= FRQ_STORE;				// frequency has been stable for a while
		}
		if(stepper_track > (R.curPos + delta_Pos))
		{
			stepper_Decr();	
			stepper_track--;
			Status |= FRQ_TIMER;				// New frequency, these flags seed timer to monitor if
			Status |= FRQ_STORE;				// frequency has been stable for a while
		}

		if (!(Menu_Mode & CONFIG) && (Menu_disp_timer==0))	// If no menu function active
		{
			lcdGotoXY(0,3);
			sprintf(lcd_buf,"Range: %2u ", range);	// Indicate which range we're using
			lcdPrintData(lcd_buf, strlen(lcd_buf));
		}
	}
}


//
//-----------------------------------------------------------------------------------------
// Top level task
// runs in an endless loop
//-----------------------------------------------------------------------------------------
//
void maintask(void)
{
	// Now we can do all kinds of business, such as measuring the AD8307 voltage output, 
	// scanning Rotary Encoder, updating LCD etc...
	static uint16_t lastIteration, lastIteration1, lastIteration2;	// Counters to keep track of time
	uint16_t Timer1val, Timer1val1, Timer1val2;		// Timers used for 100ms and 10ms polls
	static uint8_t pushcount=0;						// Measure push button time (max 2.5s)
	static uint16_t frq_store_timer;				// Keep track of when last update, in order to store
													// current status if stable

/*	char serialRx;
*/	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be run through as often as possible
	// currently measured to be approximately once every 25 - 50 us
	//-------------------------------------------------------------------------------
	encoder_Scan();								// Scan the Rotary Encoder
	
	civ_Parser();								// Read and parse frequency information from Radio
	
	#if FAST_LOOP_THRU_LED						// Blink a LED every time when going through the mainloop 
	LED_PORT = LED_PORT ^ LED;					// Blink a led
	#endif
	
	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every approx 2ms
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val1 = TCNT1/328; // get current Timer1 value, changeable every ~5ms
	//Timer1val1 = TCNT1/313;   // get current Timer1 value, changeable every ~5ms
	Timer1val1 = TCNT1/125;   // get current Timer1 value, changeable every ~2ms
//	Timer1val1 = TCNT1/62;   // get current Timer1 value, changeable every ~1ms


	if (Timer1val1 != lastIteration1)				// Once every 5ms, do stuff
	{
		lastIteration1 = Timer1val1;				// Make ready for next iteration
		#if MS_LOOP_THRU_LED						// Blink LED every 5*2 ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif

		//-------------------------------------------------------------------
		// Track frequency and adjust stepper accordingly
		//
		track_frq();

		//-------------------------------------------------------------------
		// Manually Move Stepper back and forth unless in Config Mode
		//
		if (!(Menu_Mode & CONFIG))
		{						
			if (encOutput > 0)
			{						
				delta_Pos++;						// Update position
				stepper_Incr();
				stepper_track++;
				encOutput--;
				Status |= FRQ_TIMER;				// New frequency, these flags seed timer to monitor if
				Status |= FRQ_STORE;				// frequency has been stable for a while
			}
			if (encOutput < 0)
			{	
				delta_Pos--;						// Update position
				stepper_Decr();
				stepper_track--;
				encOutput++;
				Status |= FRQ_TIMER;				// New frequency, these flags seed timer to monitor if
				Status |= FRQ_STORE;				// frequency has been stable for a while
			}
		}
	}

	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/100th of a second (10ms)
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val2 = TCNT1/656; // get current Timer1 value, changeable every ~1/100th sec
	Timer1val2 = TCNT1/626;   // get current Timer1 value, changeable every ~1/100th sec
	if (Timer1val2 != lastIteration2)				// Once every 1/100th of a second, do stuff
	{
		lastIteration2 = Timer1val2;				// Make ready for next iteration

		#if MED_LOOP_THRU_LED						// Blink LED every 10ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;  				// Blink a led
		#endif

	}

	//-------------------------------------------------------------------------------
	// Here we do routines which are to be accessed once every 1/10th of a second
	// We have a free running timer which matures once every ~1.05 seconds
	//-------------------------------------------------------------------------------
	//Timer1val = TCNT1/6554; // get current Timer1 value, changeable every ~1/10th sec
	Timer1val = TCNT1/6253;   // get current Timer1 value, changeable every ~1/10th sec
	if (Timer1val != lastIteration)	// Once every 1/10th of a second, do stuff
	{
		lastIteration = Timer1val;					// Make ready for next iteration

		#if SLOW_LOOP_THRU_LED						// Blink LED every 100ms, when going through the mainloop 
		LED_PORT = LED_PORT ^ LED;					// Blink a led
		#endif

		//-------------------------------------------------------------------
		// The Menu function has 5 seconds lag time precedence to the fourth line of the LCD
		if (Menu_disp_timer) Menu_disp_timer--;
		if (Menu_disp_timer == 1) lcdPrintData("                    ",20);	// Clear line
		
		//-------------------------------------------------------------------
		// Store Frequency in EEPROM  and power down the Stepper Motor when stable
		//
		if (Status & FRQ_STORE) 
		{

			if(Status & FRQ_TIMER)
			{
				Status &= ~FRQ_TIMER;				// Clear flag
				frq_store_timer = 0;				// Reset timer
			}
			frq_store_timer++;

			if (frq_store_timer == 100)
			{
				frq_store_timer = 0;				// Reset timer
				Status &= ~FRQ_STORE;				// Clear flag
				eeprom_write_block(&R.curPos, 
					&E.curPos, sizeof (R.curPos));
				eeprom_write_block(&R.curFrq, 
					&E.curFrq, sizeof (R.curFrq));

				stepper_PwrOff();					// Power down the stepper

			}
		}

		
		//-------------------------------------------------------------------
		// Read Pushbutton state
		//
		// Enact Long Push (pushbutton has been held down for a certain length of time):
		if (pushcount >= ENC_PUSHB_MAX)				// "Long Push", goto Configuration Mode
		{
			Menu_Mode |= CONFIG;					// Switch into Configuration Menu, while
													// retaining memory of runtime function

			Status |= LONG_PUSH;					// Used with Configuraton Menu functions	
			pushcount = 0;							// Initialize push counter for next time
		}
		// Enact Short Push (react on release if only short push)
		else if (ENC_PUSHB_INPORT & ENC_PUSHB_PIN) 	// Pin high = just released, or not pushed
		{
			// Do nothing if this is a release after Long Push
			if (Status & LONG_PUSH)					// Is this a release following a long push?
			{
					Status &= ~LONG_PUSH;			// Clear pushbutton status
			}
			// Do stuff on command
			else if (pushcount >= ENC_PUSHB_MIN)	// Check if this is more than a short spike
			{	
				if (Menu_Mode & CONFIG)
					Status |= SHORT_PUSH;			// Used with Configuraton Menu functions	

				else
				{
					//
					// Various things to be done if short push... depending on which mode is active
					//
					recalibrate_stepper_pos();
				} 
			}
			pushcount = 0;							// Initialize push counter for next time
		}
		else if (!(Status & LONG_PUSH))				// Button Pushed, count up the push timer
		{											// (unless this is tail end of a long push,
			pushcount++;							//  then do nothing)
		}

		//-------------------------------------------------------------------
		// Various Menu (rotary encoder) selectable functions
		//
		if (Menu_Mode & CONFIG)					// Pushbutton Configuration Menu
		{
			PushButtonMenu();
		}	

		else
		{
			// Timers to monitor whether we are receiving Frequency information from Radio
			static uint8_t frq_check_timer=0;	// Measures frequency receive outage time, 10x is 1 second
			static uint8_t frq_reset_timer=0;   // 5 second frequency receive poll time
			
			//
			// Poll whether we are receiving Frequency data from Transceiver
			//
			frq_reset_timer++;
			if (frq_reset_timer == 50)
			{
				Status &= ~FRQ_RADIO;			// Clear Radio Frequncy Received Status once every 5 sec
												// to check whether it is being set by the civ_parser()
				frq_reset_timer = 0;	
			}
			
			// Check if Updates from Radio are active
			//
			if ((Status & FRQ_RADIO) == 0)		// Activity flac indicates inactive
			{
				// Indicate Offline if nothing received for 15 seconds
				if (frq_check_timer == 150)
				{
					lcdGotoXY(0,0);
					lcdPrintData("Radio: Not Connected",20);
				}
				//	Else measure outage time
				else frq_check_timer++;
			}
			// Updates from Radio are active, print to LCD
			//
			else
			{
				frq_check_timer = 0; 
				lcdGotoXY(0,0);
				lcdPrintData("Radio: ",7);
				display_frq(R.curFrq);
				lcdPrintData(" Hz",3);
			}

			// Indicate Tuned Frequency
			//
			lcdGotoXY(0,1);
			lcdPrintData("Tuned: ",7);
			display_frq(tunedFrq);
			lcdPrintData(" Hz",3);

			// Indicate capacitor position (number of revolutions above lowest preset)
			//
			lcdGotoXY(0,2);
			lcdPrintData("Motor: ",7);
			display_stepper_pos(stepper_track);			
			lcdGotoXY(14,2);
			display_stepper_pos(R.curPos + delta_Pos + encOutput);

			// Indicate whether Stepper is Active or not
			if (!(Menu_Mode & CONFIG) && (Menu_disp_timer==0))	// If no menu function active
			{
				lcdGotoXY(10,3);
				if (Status & FRQ_STORE)
					lcdPrintData("Power:  On",10);
				else 
					lcdPrintData("Power: Off",10);
			}
		}

	}
	wdt_reset();								// Whoops... must remember to reset that running watchdog
}


//
//-----------------------------------------------------------------------------------------
// 			Setup Ports, timers, start the works and never return, unless reset
//								by the watchdog timer
//						then - do everything, all over again
//-----------------------------------------------------------------------------------------
//
int main(void)
{
	MCUSR &= ~(1 << WDRF);							// Disable watchdog if enabled by bootloader/fuses
	wdt_disable();

	clock_prescale_set(clock_div_1); 				// with 16MHz crystal this means CLK=16000000

	//------------------------------------------
	// 16-bit Timer1 Initialization
	TCCR1A = 0; //start the timer
	TCCR1B = (1 << CS12); // prescale Timer1 by CLK/256
	// 16000000 Hz / 256 = 62500 ticks per second
	// 16-bit = 2^16 = 65536 maximum ticks for Timer1
	// 65536 / 62500 = ~1.05 seconds
	// so Timer1 will overflow back to 0 about every 1 seconds
	// Timer1val = TCNT1; // get current Timer1 value

	//------------------------------------------
	// Init and set output for LED
	LED_DDR = LED;
	LED_PORT = 0;

	//------------------------------------------
	// Init USART RXD
	USART_DDR = USART_DDR & ~USART_RXD;	// Set pin for input

	//------------------------------------------
	// Init Pushbutton input
	ENC_PUSHB_DDR = ENC_PUSHB_DDR & ~ENC_PUSHB_PIN;	// Set pin for input
	ENC_PUSHB_PORT= ENC_PUSHB_PORT | ENC_PUSHB_PIN;	// Set pullup

	//------------------------------------------
	// Set run time parameters to Factory default under certain conditions
	//
	// Enforce "Factory default settings" when firmware is run for the very first time after
	// a fresh firmware installation with a new "serial number" in the COLDSTART_REF #define
	// This may be necessary if there is garbage in the EEPROM, preventing startup
	// To activate, roll "COLDSTART_REF" Serial Number in the PM.h file
	if (eeprom_read_byte(&E.EEPROM_init_check) != R.EEPROM_init_check)
	{
		eeprom_write_block(&R, &E, sizeof(E));		// Initialize eeprom to "factory defaults".
	}
	else
	{
		eeprom_read_block(&R, &E, sizeof(E));		// Load the persistent data from eeprom
	}

  	lcdInit();										// Init the LCD

	//------------------------------------------
	// LCD Print Version information (5 seconds)
	lcdClear();
	lcdGotoXY(0,0);
	lcdPrintData("Magnetic Loop...",16);
	_delay_ms(500);
	lcdGotoXY(3,1);
	lcdPrintData("...Controller",13);
	_delay_ms(1000);

	lcdClear();
	lcdGotoXY(0,0);
	lcdPrintData("TF3LJ / VE2LJX",14);
	lcdGotoXY(11,1);
	sprintf(lcd_buf,"V%s",VERSION);
	lcdPrintData(lcd_buf, strlen(lcd_buf));
	_delay_ms(1000);

	//lufa_init();									// Init USB-serial service

	wdt_enable(WDTO_1S);							// Start the Watchdog Timer, 1 sec

	encoder_Init();									// Init Rotary encoder
	stepper_Init();									// Init Stepper outputs
	usart_Init();									// Init USART for CI-V serial receive

	stepper_track = R.curPos;						// Init stepper value
	stepper_start_pos = R.curPos;					// Set bounds for tracking of total movement of stepper

	
	//Menu_Mode = DEFAULT_MODE;						// Power Meter Mode is normal default

	// Start the works, we're in business
	while (1)
	{
		maintask();									// Do useful stuff
		//lufa_manage_serial();						// Report to computer (if it is listening)
	}
}
